/* ----------------------------------------------------------------------- *
 *
 *   Copyright 2002 Kester Habermann - All Rights Reserved
 *   2002-08-30 Kester Habermann <kester@linuxtag.org>         
 *   Time-stamp: <2003-04-27 14:38:39 kester>
 *   $Id: knoppix-customize.c,v 0.7 2003/04/27 12:41:41 kester Exp $
 *
 *  knoppix-customize.c
 *    knoppix-customize is a program that let's you change the boot
 *    options and files of a KNOPPIX ISO image or boot disk without
 *    remastering.
 *
 *  See for latest versioN:
 *    http://hydra.hq.linuxtag.net/~kester/knoppix-customize/
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
 *   USA; either version 2 of the License, or (at your option) any later
 *   version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */


/* The code is really quite ugly as it has a lot of string processing
 * and C isn't really good at that. Of course it would have been a walk
 * in the park with perl, but the intention is to provide a small standalone
 * tool that will run anywhere (GNU/Linux, Windows, etc.)
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <getopt.h>
#include <libgen.h>
#include <limits.h>

#define KNOPPIX_CUSTOMIZE_NAME          "knoppix-customize"
#define KNOPPIX_CUSTOMIZE_AUTHOR        "Kester Habermann <kester@linuxtag.org>"
#define KNOPPIX_CUSTOMIZE_VERSION       "v0.07 (2003-04-27)"

// Sector size for FAT12 and FAT16
#define FAT_SECTOR_SIZE                  0x200
#define FLOPPY_BOOT_SECTOR               (1* FAT_SECTOR_SIZE)

#define CD_SECTOR_SIZE                   0x800
#define CD_BOOT_RECORD_VOLUME_DESCRIPTOR (1 * CD_SECTOR_SIZE)
#define CD_BOOT_CAT_BUFFER               (1 * CD_SECTOR_SIZE)
// Boot-Disc Types
// Sides * Tracks * Sectors/Track * Bytes/Block
#define BOOT_IMG_BUFFER_1_2              (2 * 15 * 80 * 512)
#define BOOT_IMG_BUFFER_1_44             (2 * 18 * 80 * 512)
#define BOOT_IMG_BUFFER_2_88             (2 * 36 * 80 * 512)

#define FLOPPY_1_2                       "1.2 MB diskette"
#define FLOPPY_1_44                      "1.44 MB diskette"
#define FLOPPY_2_88                      "2.88 MB diskette"
#define ISO_9660_IDENTIFIER              "CD001"
#define CD_BOOT_SYSTEM_INDENTIFIER       "EL TORITO SPECIFICATION"
#define MANUFACTOR_MAXLEN                 24

#define FAT_FILENAME_LENGTH               11
#define FAT_ENTRY_SIZE                    32
#define COMMAND_LINE_LENGTH               512

#define SYSLINUX_CFG_FILE                 "syslinux.cfg"

#define EXIT_OK                           0
#define EXIT_SYSLINUX_OPT_NOT_FOUND       1
#define EXIT_APPEND_OPT_NOT_FOUND         2
#define EXIT_WRONG_ARGUMENTS              3
#define EXIT_CANT_STAT_IMAGE              4
#define EXIT_CANT_STAT_LOCAL_FILE         5
#define EXIT_CANT_READ_IMAGE              6
#define EXIT_CANT_PARSE_IMAGE             7
#define EXIT_CANT_EXPORT                  8
#define EXIT_CANT_IMPORT                  9
#define EXIT_TARGET_NOT_FOUND            10
#define EXIT_CANT_READ_OPTIONS           11
#define EXIT_NO_VALUE_EXPECTED           12
#define EXIT_NOT_IMPLEMENTED            200
#define EXIT_INTERNAL_ERROR             201

// Some macros
#define VERB_PRINT(x) if (opt_verbose) fprintf(stderr, x)
#define VERB_PRINT2(x, y) if (opt_verbose) fprintf(stderr, x, y)
#define VERB_PRINT3(x, y, z) if (opt_verbose) fprintf(stderr, x, y, z)
#define VERB_PRINT4(x, y, z, t) if (opt_verbose) fprintf(stderr, x, y, z, t)

// taken from /usr/include/glib-2.0/glib/gutils.h
#ifdef G_OS_WIN32

/* On native Win32, directory separator is the backslash, and search path
 * separator is the semicolon.
 */
#define G_DIR_SEPARATOR '\\'
#define G_DIR_SEPARATOR_S "\\"
#define G_SEARCHPATH_SEPARATOR ';'
#define G_SEARCHPATH_SEPARATOR_S ";"

#else  /* !G_OS_WIN32 */

/* Unix */

#define G_DIR_SEPARATOR '/'
#define G_DIR_SEPARATOR_S "/"
#define G_SEARCHPATH_SEPARATOR ':'
#define G_SEARCHPATH_SEPARATOR_S ":"

#endif /* !G_OS_WIN32 */


// one directory entry
struct dir_entry_t {
    unsigned char name[FAT_FILENAME_LENGTH + 1];
    unsigned long pos;
    size_t size;
};

// important data from boot floppy
struct boot_floppy_t {
    unsigned long boot_img_offset;
    size_t boot_img_size;
    unsigned int FirstRootDirSecNum;
    unsigned int BytesPerCluster;
    struct dir_entry_t *fat_root_dir;
    unsigned int fat_root_dir_size;
};

// Global variables
int opt_verbose = 0;

int find_boot_image(FILE *fh, struct stat stat_buf, struct boot_floppy_t *boot_floppy) {
    unsigned char *floppy_boot_sector;    // buffer for boot record volume descriptor
    unsigned char *cd_boot_rvd;           // buffer for boot record volume descriptor
    unsigned char *cd_boot_cat;           // buffer for boot catalog
    char cd_manufactor[MANUFACTOR_MAXLEN];
    unsigned long cd_boot_cat_ptr;
    unsigned long cd_boot_img_ptr;

    // allocate read_buffer for reading boot record volume descriptor
    if ((cd_boot_rvd = malloc((size_t) (CD_BOOT_RECORD_VOLUME_DESCRIPTOR))) == NULL) {
        fprintf(stderr, "Error mallocing CD boot record volume descriptor\n");
        return 0;
    }

    // allocate read_buffer for reading boot catalog
    if ((cd_boot_cat = malloc((size_t) (CD_BOOT_CAT_BUFFER))) == NULL) {
        fprintf(stderr, "Error mallocing CD boot cat buffer\n");
        return 0;
    }

    // allocate read_buffer for reading floppy boot sector
    if ((floppy_boot_sector = malloc((size_t) (FLOPPY_BOOT_SECTOR))) == NULL) {
        fprintf(stderr, "Error mallocing floppy boot sector buffer\n");
        return 0;
    }

    // read first sectors of size FAT_SECTOR, to see if it's FAT
    if (fread(((void *) floppy_boot_sector),
              (size_t) FLOPPY_BOOT_SECTOR,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error reading first sector from image\n");
        return 0;
    }

    // Test whether image ist FAT, this is the case if someone chooses
    // to only modify a boot disk and not the whole book image
    // Search for FAT-Magic 0x55AA

    if (floppy_boot_sector[0x1fe] == (unsigned char) 0x55 && floppy_boot_sector[0x1ff] == (unsigned char) 0xAA) {
        VERB_PRINT("Image is FAT, assuming floppy\n");
        boot_floppy->boot_img_offset = 0;
        // find out actual size of disk image
        switch (stat_buf.st_size) {
            case BOOT_IMG_BUFFER_1_2:  VERB_PRINT2("Image is %s\n", FLOPPY_1_2);
                                       boot_floppy->boot_img_size = BOOT_IMG_BUFFER_1_2;
                                       break;

            case BOOT_IMG_BUFFER_1_44: VERB_PRINT2("Image is %s\n", FLOPPY_1_44);
                                       boot_floppy->boot_img_size = BOOT_IMG_BUFFER_1_44;
                                       break;

            case BOOT_IMG_BUFFER_2_88: VERB_PRINT2("Image is %s\n", FLOPPY_2_88);
                                       boot_floppy->boot_img_size = BOOT_IMG_BUFFER_2_88;
                                       break;

            default:                   fprintf(stderr, "Invalid disk image size (%d)\n", (int) stat_buf.st_size);
                                       return 0;
        }
        return 1;
    } 

    //
    // Read CD-Image
    //
    VERB_PRINT("Image is not FAT, assuming ISO\n");
    // read boot volume descriptor
    if (fseek(fh, 0x11 * CD_SECTOR_SIZE, SEEK_SET)) {
        fprintf(stderr, "Error seeking into file\n");
        return 0;
    }
    
    if (fread(((void *) cd_boot_rvd),
              (size_t) CD_BOOT_RECORD_VOLUME_DESCRIPTOR,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error reading cd boot volume descriptor\n");
        return 0;
    }
    
    if (cd_boot_rvd[0x00] != (unsigned char) 0) {
        fprintf(stderr, "Invalid CD Boot Record Indicator\n");
        return 0;
    }
    
    if ( (strncmp((char *) cd_boot_rvd + 0x01, ISO_9660_IDENTIFIER, (size_t) sizeof(ISO_9660_IDENTIFIER) -1)) != 0) {
        fprintf(stderr, "Not ISO 9660 image\n");
        return 0;
    }
    
    if ( (strncmp((char *) cd_boot_rvd + 0x07, CD_BOOT_SYSTEM_INDENTIFIER, (size_t) sizeof(CD_BOOT_SYSTEM_INDENTIFIER) -1)) != 0) {
        fprintf(stderr, "Not El Torito Bootable\n");
        return 0;
    }  
    
    cd_boot_cat_ptr = (unsigned long) ( cd_boot_rvd[0x47 + 3] << 24 )
		    + (unsigned long) ( cd_boot_rvd[0x47 + 2] << 16 )
		    + (unsigned long) ( cd_boot_rvd[0x47 + 1] <<  8 )
		    + (unsigned long) ( cd_boot_rvd[0x47 + 0]       );

    free(cd_boot_rvd);

    VERB_PRINT3("boot cat ptr: 0x%.8lx x 0x800 (0x%.8lx)\n", cd_boot_cat_ptr, CD_SECTOR_SIZE * cd_boot_cat_ptr);
    
    // seek to boot catalog
    if (fseek(fh, CD_SECTOR_SIZE * cd_boot_cat_ptr, SEEK_SET)) {
        fprintf(stderr, "Error seeking to CD boot cat\n\n");
        return 0;
    }
    
    if (fread(((void *) cd_boot_cat),
	      (size_t) CD_BOOT_CAT_BUFFER,
	      (size_t) 1,
	      fh) < 1) {
	fprintf(stderr, "Error reading CD boot cat\n");
        return 0;
    }
    
    if (cd_boot_cat[0] != (unsigned char) 01) {
        fprintf(stderr, "Header ID, must be 01\n");
        return 0;
    }
    
    switch(cd_boot_cat[1]) {
        case 0:  VERB_PRINT("Platform 80x86\n");
                 break;
        case 1:  VERB_PRINT("Platform Power PC\n");
                 break;
        case 2:  VERB_PRINT("Platform MAC\n");
                 break;
        default: fprintf(stderr, "Platform unknown\n");
                 return 0;
    }
    
    strncpy(cd_manufactor, (char *) cd_boot_cat + 0x04, MANUFACTOR_MAXLEN);
    VERB_PRINT2("Manufactor: %s\n", cd_manufactor);
    
    switch (cd_boot_cat[0x20 + 0x00]) {
        case 0x00: fprintf(stderr, "CD Not Bootable!\n");
                   return 0;
        case 0x88: VERB_PRINT("CD Bootable!\n");
                   break;
        default:   fprintf(stderr, "Invalid Boot Indicator on CD\n");
                   return 0;
    }

    VERB_PRINT("Boot media type on CD: ");
    switch (cd_boot_cat[0x20 + 0x01]) {
        case 0x00: fprintf(stderr, "No emulation (can't handle this)\n");
                   return 0;
        case 0x01: VERB_PRINT2("%s\n", FLOPPY_1_2);
                   boot_floppy->boot_img_size = BOOT_IMG_BUFFER_1_2;
                   break;
        case 0x02: VERB_PRINT2("%s\n", FLOPPY_1_44);
                   boot_floppy->boot_img_size = BOOT_IMG_BUFFER_1_44;
                   break;
        case 0x03: VERB_PRINT2("%s\n", FLOPPY_2_88);
                   boot_floppy->boot_img_size = BOOT_IMG_BUFFER_2_88;
                   break;
        case 0x04: fprintf(stderr, "Harddisk (can't handle this)\n");
                   return 0;
        default:   fprintf(stderr, "Invalid Boot media type\n");
                   return 0;
    }

    cd_boot_img_ptr =   (unsigned int) ( cd_boot_cat[0x20 + 0x08 + 3] << 24 )
                      + (unsigned int) ( cd_boot_cat[0x20 + 0x08 + 2] << 16 )
                      + (unsigned int) ( cd_boot_cat[0x20 + 0x08 + 1] <<  8 )
                      + (unsigned int) ( cd_boot_cat[0x20 + 0x08 + 0]       );

    boot_floppy->boot_img_offset = CD_SECTOR_SIZE * cd_boot_img_ptr;

    VERB_PRINT3("boot img ptr: 0x%.8lx x 0x800 (0x%.8lx)\n", cd_boot_img_ptr, boot_floppy->boot_img_offset);

    return 1;
}

int get_fat_root_dir (FILE *fh, struct boot_floppy_t *boot_floppy) {
    unsigned char *boot_img;              // buffer for boot image
    unsigned char *floppy_boot_sector;    // buffer for boot record volume descriptor
    unsigned int BPB_BytsPerSec;
    unsigned char BPB_SecPerClus;
    unsigned int BPB_RsvdSecCnt;
    unsigned char BPB_NumFATs;
    unsigned int BPB_FATSz16;
    unsigned char *fat_root_dir_buffer;
    unsigned int fat_root_dir_buffer_size;
    int i, j;

    //
    // Read Floppy-Image
    //

    // allocate read_buffer for reading boot image
    if ((boot_img = malloc(boot_floppy->boot_img_size)) == NULL) {
        fprintf(stderr, "Error mallocing CD boot img buffer\n");
        return 0;
    }

    VERB_PRINT("\nReading boot image ...\n");
    // seek to boot image
    if (fseek(fh, (long) boot_floppy->boot_img_offset, SEEK_SET)) {
        fprintf(stderr, "Error seeking to boot image\n");
	return 0;
    }

    // allocate read_buffer for reading floppy boot sector
    if ((floppy_boot_sector = malloc((size_t) (FLOPPY_BOOT_SECTOR))) == NULL) {
        fprintf(stderr, "Error mallocing floppy boot sector buffer\n");
        return 0;
    }

    // read first floppy boot sector
    if (fread(((void *) floppy_boot_sector),
              (size_t) FLOPPY_BOOT_SECTOR,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error reading boot sector from image\n");
        return 0;
    }

    BPB_BytsPerSec =   (unsigned int) ( floppy_boot_sector[12] << 8 )
                     + (unsigned int) ( floppy_boot_sector[11]      );

    BPB_SecPerClus =                    floppy_boot_sector[13];

    boot_floppy->BytesPerCluster = (unsigned int) BPB_SecPerClus * BPB_BytsPerSec;

    BPB_RsvdSecCnt =   (unsigned int) ( floppy_boot_sector[15] << 8 )
                     + (unsigned int) ( floppy_boot_sector[14]      );

    BPB_NumFATs    =                    floppy_boot_sector[16];

    BPB_FATSz16    =   (unsigned int) ( floppy_boot_sector[23] << 8 )
                     + (unsigned int) ( floppy_boot_sector[22]      );

    fat_root_dir_buffer_size  = (unsigned int) BPB_NumFATs * BPB_BytsPerSec;
    boot_floppy->FirstRootDirSecNum = BPB_RsvdSecCnt + ( (unsigned int) BPB_NumFATs * BPB_FATSz16);

    VERB_PRINT2("BPB_BytsPerSec:     %d\n", BPB_BytsPerSec);
    VERB_PRINT2("BPB_SecPerClus:     %d\n", (int) BPB_SecPerClus);
    VERB_PRINT2("BPB_RsvdSecCnt:     %d\n", BPB_RsvdSecCnt);
    VERB_PRINT2("BPB_NumFATs:        %d\n", (int) BPB_NumFATs);
    VERB_PRINT2("BPB_FATSz16:        %d\n", BPB_FATSz16);

    VERB_PRINT("\n");

    VERB_PRINT2("FAT Root Dir Buffer Size:  %d\n", fat_root_dir_buffer_size);
    VERB_PRINT2("FirstRootDirSecNum:        %d\n", boot_floppy->FirstRootDirSecNum);

    VERB_PRINT("\n");

    // allocate buffer for FAT root dir
    if ((fat_root_dir_buffer = malloc((size_t) (fat_root_dir_buffer_size))) == NULL) {
        fprintf(stderr, "Error mallocing for FAT root-Rootdir\n");
        return 0;
    }

    // seek to FirstRootDirSecNum
    if (fseek(fh, boot_floppy->boot_img_offset + (unsigned long) (boot_floppy->FirstRootDirSecNum * BPB_BytsPerSec), SEEK_SET) != 0) {
        fprintf(stderr, "Error seeking into file\n");
        return 0;
    }
    
    // read FirstRootDirSecNum
    if (fread(((void *) fat_root_dir_buffer),
              (size_t) fat_root_dir_buffer_size,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error fat root dir\n");
        return 0;
    }

    // Count directory entries
    boot_floppy->fat_root_dir_size = 0;
    for (i = 0; i <= fat_root_dir_buffer_size - FAT_ENTRY_SIZE; i += FAT_ENTRY_SIZE) {
	// empty entry
	if (fat_root_dir_buffer[i] == (unsigned char) 0xE5)
	    continue;

	// hidden entry
	if ((fat_root_dir_buffer[i + 11] & (unsigned char) 0x0f) != (unsigned char) 0)
	    continue;

	// no more entries
	if (fat_root_dir_buffer[i] == (unsigned char) 0x00)
	    break;

	boot_floppy->fat_root_dir_size++;
    }

    VERB_PRINT2("FAT root dir size:  %d\n", boot_floppy->fat_root_dir_size);

    // allocate buffer for FAT root dir
    if ((boot_floppy->fat_root_dir = malloc((size_t) (boot_floppy->fat_root_dir_size * FAT_ENTRY_SIZE))) == NULL) {
        fprintf(stderr, "Error mallocing for FAT root-Rootdir\n");
        return 0;
    }

    j = 0;
    for (i = 0; i <= fat_root_dir_buffer_size - FAT_ENTRY_SIZE; i += FAT_ENTRY_SIZE) {
	// empty entry
	if (fat_root_dir_buffer[i] == (unsigned char) 0xE5)
	    continue;

	// hidden entry
	if ((fat_root_dir_buffer[i + 11] & (unsigned char) 0x0f) != (unsigned char) 0)
	    continue;

	// no more entries
	if (fat_root_dir_buffer[i] == (unsigned char) 0x00)
	    break;

	strncpy(boot_floppy->fat_root_dir[j].name, fat_root_dir_buffer + i, (size_t) FAT_FILENAME_LENGTH);
	boot_floppy->fat_root_dir[j].name[FAT_FILENAME_LENGTH] = '\0';

	boot_floppy->fat_root_dir[j].pos  =  (unsigned long) ( fat_root_dir_buffer[i + 21] << 24 )
	                                                   + ( fat_root_dir_buffer[i + 20] << 16 )
							   + ( fat_root_dir_buffer[i + 27] <<  8 )
							   + ( fat_root_dir_buffer[i + 26]       );
	    
	boot_floppy->fat_root_dir[j].size =   (size_t)( fat_root_dir_buffer[i + 31] << 24 )
						    + ( fat_root_dir_buffer[i + 30] << 16 )
						    + ( fat_root_dir_buffer[i + 29] <<  8 )
						    + ( fat_root_dir_buffer[i + 28]       );
	j++;
    }

    free(fat_root_dir_buffer);

    return 1;
}

// convert file name to fat file name
void filename2fat(unsigned char *dst, const unsigned char *src) {
    int i, j;

    // convert searched filename to fat filename
    // - uppercase
    // - remove dot
    // - fill gap and end with spaces 

    j = 0;
    for (i = 0; i <= strlen(src) - 1; i++) {
	if (src[i] == '.') {
	    while(j <= FAT_FILENAME_LENGTH - 4) {
		dst[j] = ' ';
		j++;
	    }
	} else {
	    dst[j] = toupper(src[i]);
	    j++;
	}
    }
    
    while(j <= FAT_FILENAME_LENGTH - 1) {
	dst[j] = ' ';
	j++;
    }

    dst[FAT_FILENAME_LENGTH] = '\0';
}

void fat2filename(unsigned char *dst, const unsigned char *src) {
    size_t i, j;

    for (i = 0; i <=  strlen(src); i++) {
	if (src[i] == ' ' || src[i] == '\0' || i >= FAT_FILENAME_LENGTH - 3)
	    break;
	dst[i] = tolower(src[i]);
    }
    
    j = FAT_FILENAME_LENGTH - 3;

    if (src[j] != ' ' && src[j] != '\0') {
	dst[i++] = '.';
	for (; j <= strlen(src); j++) {
	    if (src[j] == ' ' || src[j] == '\0' || j >= FAT_FILENAME_LENGTH)
		break;
	    dst[i++] = tolower(src[j]);
	}
    } 
    dst[i] = '\0';
}

int file_exists_in_image(const unsigned char *fname, struct boot_floppy_t *boot_floppy) {
    int i;
    int found = 0;
    unsigned char fatname[FAT_FILENAME_LENGTH + 2];

    filename2fat(fatname, fname);

    for (i = 0; i <= boot_floppy->fat_root_dir_size - 1; i++) {
	if (strncmp(boot_floppy->fat_root_dir[i].name, fatname, FAT_FILENAME_LENGTH) == 0) {
	    found = 1;
	    break;
	}
    }
    
    return found;
}

void show_root_dir(struct boot_floppy_t *boot_floppy) {
    int i;

    for (i = 0; i <= boot_floppy->fat_root_dir_size - 1; i++) {
	VERB_PRINT4("File: '%s' Pos: %8ld, Size: %8ld\n",
				boot_floppy->fat_root_dir[i].name,
				boot_floppy->fat_root_dir[i].pos,
				(long int) boot_floppy->fat_root_dir[i].size);
    }
}

void list_files(struct boot_floppy_t *boot_floppy) {
    int i;
    unsigned char fname[FAT_FILENAME_LENGTH + 2];

    for (i = 0; i <= boot_floppy->fat_root_dir_size - 1; i++) {
	fat2filename(fname, boot_floppy->fat_root_dir[i].name);
	printf("%s\n", fname);
    }
}

struct dir_entry_t get_dir_entry(const unsigned char *fname, struct boot_floppy_t *boot_floppy) {
    unsigned char fat_name[FAT_FILENAME_LENGTH + 2];
    int i;
    struct dir_entry_t dir_entry;

    filename2fat(fat_name, fname);

    for (i = 0; i <= boot_floppy->fat_root_dir_size - 1; i++) {
	if (strncmp(boot_floppy->fat_root_dir[i].name, fat_name, FAT_FILENAME_LENGTH) == 0) {
	    strncpy(dir_entry.name, boot_floppy->fat_root_dir[i].name, FAT_FILENAME_LENGTH);
	    dir_entry.pos  = boot_floppy->fat_root_dir[i].pos;
	    dir_entry.size = boot_floppy->fat_root_dir[i].size;
	    break;
	}
    }

    VERB_PRINT4("File: [%s], Pos: %ld, Size: %ld\n", fname, dir_entry.pos, (long int) dir_entry.size);

    return dir_entry;
}

int read_file(FILE *fh, struct boot_floppy_t *boot_floppy, struct dir_entry_t dir_entry, unsigned char *buffer) {
    // seek to file
    if (fseek(fh,
	      (unsigned long) (boot_floppy->boot_img_offset + (dir_entry.pos + boot_floppy->FirstRootDirSecNum - 1) * boot_floppy->BytesPerCluster),
	      SEEK_SET) != 0) {
        fprintf(stderr, "Error seeking into file\n");
        return 0;
    }
    
    // read file into buffer
    if (fread(((void *) buffer),
              (size_t) dir_entry.size,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error reading file\n");
        return 0;
    }

    return 1;
}

int write_file(FILE *fh, struct boot_floppy_t *boot_floppy, struct dir_entry_t dir_entry, const unsigned char *buffer) {
    // seek to file
    if (fseek(fh,
	      (unsigned long) (boot_floppy->boot_img_offset + (dir_entry.pos + boot_floppy->FirstRootDirSecNum - 1) * boot_floppy->BytesPerCluster),
	      SEEK_SET) != 0) {
        fprintf(stderr, "Error seeking into file\n");
        return 0;
    }
    
    // read file into buffer
    if (fwrite(((void *) buffer),
              (size_t) dir_entry.size,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error writing file\n");
        return 0;
    }

    return 1;
}

void show_buffer(const unsigned char *buffer, size_t size) {
    size_t i;

    VERB_PRINT(">>>>");
    for (i = 0; i <= size - 1; i++)
	VERB_PRINT2("%c", buffer[i]);
    VERB_PRINT("<<<<\n");
}

int output_buffer(FILE *fh, unsigned char *buffer, size_t size) {
    if (fwrite(((void *) buffer),
	       (size_t) size,
	       (size_t) 1,
	       fh) < 1) {
        fprintf(stderr, "Error outputting file\n");
    }

    return 1;
}

int export_to_local_file (FILE *fh, struct boot_floppy_t *boot_floppy, const unsigned char *image_file, const unsigned char *local_file) {
    FILE *fh2;
    unsigned char *buffer;
    struct dir_entry_t dir_entry;

    dir_entry = get_dir_entry(image_file, boot_floppy);
    
    // allocate read_buffer for reading file
    if ((buffer = malloc((size_t) (dir_entry.size))) == NULL) {
	fprintf(stderr, "Error mallocing buffer for reading file\n");
	return 0;
    }
    
    if (read_file(fh, boot_floppy, dir_entry, buffer) == 0) {
	fprintf(stderr, "Error reading file. Aborting\n");
	return 0;
    }
    
    if (strncmp(local_file, "-", 1) == 0)
	fh2 = stdout;
    else
	if ((fh2 = fopen(local_file, "wb")) == NULL) {
	    fprintf(stderr, "Error writing file '%s'\n", local_file);
	    return 0;
	}
    
    if(output_buffer(fh2, buffer, dir_entry.size) == 0) {
	fprintf(stderr, "Error outputting buffer. Aborting\n");
	(void) fclose(fh2);
	return 0;
    }
    
    (void) fclose(fh2);
    free(buffer);

    return 1;
}

int import_from_local_file(FILE *fh, struct boot_floppy_t *boot_floppy, const unsigned char *image_file, const unsigned char *local_file) {
    FILE *fh2;
    unsigned char *buffer;
    struct dir_entry_t dir_entry;

    dir_entry = get_dir_entry(image_file, boot_floppy);
    
    // allocate read_buffer for reading file
    if ((buffer = malloc((size_t) (dir_entry.size))) == NULL) {
	fprintf(stderr, "Error mallocing buffer for reading file\n");
	return 0;
    }

    if (strncmp(local_file, "-", 1) == 0)
	fh2 = stdin;
    else
	if ((fh2 = fopen(local_file, "rb")) == NULL) {
	    fprintf(stderr, "Error reading file '%s'\n", local_file);
	    return 0;
	}
    
    // read local file into buffer
    if (fread(((void *) buffer),
              (size_t) dir_entry.size,
              (size_t) 1,
              fh2) < 1) {
        fprintf(stderr, "Error reading local file\n");
        return 0;
    }

    if (write_file(fh, boot_floppy, dir_entry, buffer) == 0) {
	fprintf(stderr, "Error writing file. Aborting\n");
	return 0;
    }

    (void) fclose(fh2);
    free(buffer);

    return 1;
}

int import_floppy(FILE *fh, struct boot_floppy_t *boot_floppy, const unsigned char *local_file) {
    FILE *fh2;
    unsigned char *buffer;

    // allocate read_buffer for reading file
    if ((buffer = malloc((size_t) (boot_floppy->boot_img_size))) == NULL) {
	fprintf(stderr, "Error mallocing buffer for reading file\n");
	return 0;
    }

    if (strncmp(local_file, "-", 1) == 0)
	fh2 = stdin;
    else
	if ((fh2 = fopen(local_file, "rb")) == NULL) {
	    fprintf(stderr, "Error reading file '%s'\n", local_file);
	    return 0;
	}
    
    // read local file into buffer
    if (fread(((void *) buffer),
              (size_t) boot_floppy->boot_img_size,
              (size_t) 1,
              fh2) < 1) {
        fprintf(stderr, "Error reading local file\n");
        return 0;
    }

    // seek to boot image
    if (fseek(fh, boot_floppy->boot_img_offset, SEEK_SET)) {
        fprintf(stderr, "Error seeking to image\n");
        return 0;
    }

    if (fwrite(((void *) buffer),
	       (size_t) boot_floppy->boot_img_size,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error writing to image\n");
        return 0;
    }

    (void) fclose(fh2);
    free(buffer);

    return 1;
}

int export_floppy(FILE *fh, struct boot_floppy_t *boot_floppy, const unsigned char *local_file) {
    FILE *fh2;
    unsigned char *buffer;
    
    // allocate read_buffer for reading file
    if ((buffer = malloc((size_t) (boot_floppy->boot_img_size))) == NULL) {
	fprintf(stderr, "Error mallocing buffer for reading file\n");
	return 0;
    }

    // seek to boot image
    if (fseek(fh, boot_floppy->boot_img_offset, SEEK_SET)) {
        fprintf(stderr, "Error seeking to boot image\n");
        return 0;
    }

    // read local file into buffer
    if (fread(((void *) buffer),
              (size_t)  boot_floppy->boot_img_size,
              (size_t) 1,
              fh) < 1) {
        fprintf(stderr, "Error reading boot floppy from image\n");
        return 0;
    }
    
    if (strncmp(local_file, "-", 1) == 0)
	fh2 = stdout;
    else
	if ((fh2 = fopen(local_file, "wb")) == NULL) {
	    fprintf(stderr, "Error writing file '%s'\n", local_file);
	    return 0;
	}
    
    if(output_buffer(fh2, buffer, boot_floppy->boot_img_size) == 0) {
	fprintf(stderr, "Error outputting buffer. Aborting\n");
	fclose(fh2);
	return 0;
    }
    
    fclose(fh2);
    free(buffer);

    return 1;
}

// read command line up to newline
int read_commandline(unsigned char *commandline) {
    int c, i = 0, status = 1;

    while( (c=getchar()) != EOF ) {
	if (c == '\n') {
	    break;
	}

	if (i >= COMMAND_LINE_LENGTH) {
	    status = 0;
	    break;
	}

	commandline[i++] = c;
    }

    commandline[i] = '\0';

    return status;
}

// s/^\s*//; s/\*$//;
void trim_spaces(unsigned char *string) {
    unsigned char *start, *end, *i;
    int len, j;

    len   = strlen(string);
    start = string;
    end   = start + len;

    for (j = 0; j < strlen(string); j++)
	if (string[j] != ' ')
	    break;
    
    start = string + j;

    for (j = len - 1; j >= 0; j--)
	if (string[j] != ' ')
	    break;
    
    end = string + j;
    
    j = 0;
    for (i = start; i <= end; i++)
	string[j++] = *i;
    
    string[j] = '\0';
}

// parse command line: seperate keyword and value
// returns
//   0 on error
//   1 only keyword given
//   2 keyword and value given
int parse_command(const unsigned char *commandline, unsigned char *keyword, unsigned char *value) {
    unsigned char *pos;
    int status = 0;

    pos = strchr(commandline, '=');

    // only keyword
    if (pos == NULL) {
	strncpy(keyword, commandline, COMMAND_LINE_LENGTH);
	value[0] = '\0';
	status = 1;
    } else {
	strncpy(keyword, commandline, pos - commandline);
	strncpy(value, pos + 1, COMMAND_LINE_LENGTH);

	status = 2;
    }

    return status;
}

// compate to strings case insensitive
int mystrnicmp(const unsigned char *s, const unsigned char *t, int n) {
    int i = 0;

    for ( ; tolower(*s) == tolower(*t) && i++ <= n; s++, t++)
	if (*s == '\0' || i == n)
	    return 0;
    return tolower(*s) - tolower(*t);
}

// find_syslinux_opt
//   opt not found:   returns 0
//   opt found:       returns 1
int find_syslinux_opt(const unsigned char *keyword, const unsigned char *buffer, int size, unsigned char **pos, int *len) {
    int found = 0;
    int i, start = 0;

    for (i = 0; i < size; i++) {
	if (i > 0)
	    // only match at beginning of line
	    if (buffer[i -1] != '\n' && buffer[i -1] != '\r')
		continue;

	if (mystrnicmp(keyword, buffer + i, strlen(keyword)) == 0) {
	    found = 1;
	    start = i + strlen(keyword);
	    *pos = (unsigned char*) buffer + start;
	    break;
	}
    }

    if (found) {
	i = 0;
	while ((*pos)[i] != '\n' && (*pos)[i] != '\0' && start + i < size)
	    i++;

	*len = i;
    }

    return found;
}


// find_target
//   target found     returns 1
//   target not found returns 0
int find_target(int opt_target, const unsigned char *target, const unsigned char *buffer,
		int size, unsigned char **pos, int *len) {
    int found = 0;
    unsigned char search[COMMAND_LINE_LENGTH +1 ];
    int len2;
    unsigned char* pos2;
    
    *pos = (unsigned char*) buffer;
    
    // if target given and not default, find label
    if (opt_target && mystrnicmp(target, "default", 7) != 0) {
	strncpy(search, "LABEL ", 7);

	strncat(search, target, COMMAND_LINE_LENGTH - strlen(search));

	found = find_syslinux_opt(search, buffer, size, pos, len);
	if (found)
	    *len = size - (*pos - buffer);
    } else {
	*len = size;
	found = 1;
    }
    
    // now we have beginning of target find next label for end
    strncpy(search, "LABEL ", 7);
	
    if (find_syslinux_opt(search, *pos + 1, *len, &pos2, &len2)) {
	*len = pos2 - *pos - 6;
    }

    return found;
}

// get_syslinux_opt
//   opt not found:   returns 0
//   opt found:       returns 1
int get_syslinux_opt(const unsigned char *keyword, const unsigned char *buffer, int size) {
    int found = 0;
    int len, i;
    unsigned char *pos = NULL;
    unsigned char search[COMMAND_LINE_LENGTH +1 ];

    // add space after keyword so we may not match substrings
    strncpy(search, keyword, COMMAND_LINE_LENGTH);
    strncat(search, " ", 1);

    found = find_syslinux_opt(search, buffer, size, &pos, &len);

    if (found) {
	for (i = 0; i < len; i++) 
	    printf("%c", pos[i]); 
	printf("\n");
    }
    
    return found;
}

// returns 0 on error
//         1 on success
int make_space(unsigned char **buffer, int size, const unsigned char *pos, int len) {
    int i;
    unsigned char *buf;

    buf = *buffer;

    if ((pos + len > buf + size - 2) || (pos + len < buf)) {
	fprintf(stderr, "Error: make_space out of bounds\n");
	return 0;
    }
	    
    if (len > 0) {
	for (i =  size - len - 2; i >= pos - buf + 2; i--)
	    buf[i + len] = buf[i];	
    } else if (len < 0) {
	for (i =  pos - buf + len ; i < size - 1; i++)
	    buf[i] = buf[i - len];
	
	for (i =  buf + size + len - buf - 1; i < size - 1; i++)
	    buf[i] = '#';
    } // nothing to do for len == 0

    return 1;
}

// set_syslinux_opt
//   set            returns 1
//   error setting  returns 0
int set_syslinux_opt(FILE *fh, struct boot_floppy_t *boot_floppy,
		     const unsigned char *keyword, const unsigned char *value, int remove_opt,
		     unsigned char *buffer, int size,
		     const unsigned char *pos, int len) {
    int found = 0;
    int len2, i;
    unsigned char *pos2 = NULL;
    unsigned char search[COMMAND_LINE_LENGTH +1 ];
    unsigned char old_val[COMMAND_LINE_LENGTH +1 ];
    struct dir_entry_t dir_entry;
    int space_needed = 0;

    // add space after keyword so we may not match substrings
    strncpy(search, keyword, COMMAND_LINE_LENGTH);
    strncat(search, " ", 1);

    found = find_syslinux_opt(search, pos, len, &pos2, &len2);

    if (found) {
	strncpy(old_val, pos2, len2);

	old_val[len2] = '\0';

	if (remove_opt) {
	    space_needed = - ( strlen(keyword) + 2 + len2 );

	    show_buffer(buffer, size);

	    make_space(&buffer, size, pos2 - strlen(keyword) - 1 - space_needed, space_needed);
	} else {
	    if (strlen(value) != len2) {
		space_needed = strlen(value) - len2;
		make_space(&buffer, size, pos2 + 1 - space_needed, space_needed);
	    } 	    

	    VERB_PRINT4("Setting value for syslinux option '%s' from '%s' to '%s'\n", keyword, old_val, value);
	    for (i = 0; i < strlen(value); i++) 
		pos2[i] = value[i];
	    
/*  	    if (strlen(value) != len2) */
/*  		pos2[i] = '\n'; */

	}
	
	if (pos2[i] != '\n')
	    pos2[i] = '\n';
	
	VERB_PRINT("Result:\n");
	show_buffer(buffer, size);
	
	dir_entry = get_dir_entry(SYSLINUX_CFG_FILE, boot_floppy);
	
	if (write_file(fh, boot_floppy, dir_entry, buffer) == 0) {
		    fprintf(stderr, "Error writing file. Aborting\n");
	    return 0;
	}
    } else {
	VERB_PRINT("option not found\n");
	found = 0;
    }
    
    return found;   
}

// find_append_opt
//   opt not found:   returns 0
//   opt found:       returns 1
int find_append_opt(const unsigned char *keyword, const unsigned char *buffer, int size, unsigned char **pos, int *len) {
    int found = 0;
    int i;
    unsigned char *ppos;
    
    if (find_syslinux_opt("APPEND ", buffer, size, pos, len) != 0) {
	ppos = *pos;
	for (i = 0; i < *len; i++) {   
	    if (mystrnicmp(keyword, ppos + i, strlen(keyword)) == 0) {
		if (ppos[i+strlen(keyword)] == '=' || ppos[i+strlen(keyword)] == ' ' || ppos[i+strlen(keyword)] == '\n') {
		    found = 1;
		    
		    ppos = ppos + i;
		    
		    break;
		}
	    }
	}
	
	
	if (found) {
	    i = 0;
	    
	    while (ppos[i] != '\n' && ppos[i] != ' ' && ppos[i] != '\0' && ppos + i < buffer + size)
		i++;
	    
	    *len = i;

	    *pos = ppos;
		
	}
	
    }

    return found;
}

// get_append_opt
//   opt not found:   returns 0
//   opt found:       returns 1
int get_append_opt(const unsigned char *keyword, const unsigned char *buffer, int size) {
    int found = 0;
    int len, i;
    unsigned char *pos = NULL;
    unsigned char search[COMMAND_LINE_LENGTH +1 ];

    strncpy(search, keyword, COMMAND_LINE_LENGTH);

    found = find_append_opt(search, buffer, size, &pos, &len);
    
    VERB_PRINT2("len:         %d\n", len);
    VERB_PRINT2("len keyword: %d\n", strlen(keyword));
    VERB_PRINT2("keyword:     %s\n", keyword);

    pos += strlen(keyword);
    if (pos[0] == '=') {
	pos += 1;

	if (found && len > 0) {
	    for (i = 0; i < len - strlen(keyword) - 1; i++) 
		printf("%c", pos[i]); 
	    printf("\n");
	}
    } 
    
    return found;
}


int set_append_opt(FILE *fh, struct boot_floppy_t *boot_floppy,
		     const unsigned char *keyword, const unsigned char *value, int remove_opt,
		     unsigned char *buffer, int size,
		   const unsigned char *pos, int len) {
    
    int found = 0;
    int len2, i;
    unsigned char *pos2 = NULL;
    unsigned char search[COMMAND_LINE_LENGTH +1 ];
    unsigned char old_val[COMMAND_LINE_LENGTH +1 ];
    struct dir_entry_t dir_entry;
    int space_needed = 0;

    // add = after keyword so we may not match substrings
    strncpy(search, keyword, COMMAND_LINE_LENGTH);

    found = find_append_opt(search, pos, len, &pos2, &len2);
    len2 = len2 - strlen(keyword) - 1;

    if (found) {
	pos2 += strlen(keyword);
	if (pos2[0] == '=') {
	    pos2 += 1;
	}
    }

    if (remove_opt) {
	if (found) {
	    space_needed = - ( strlen(keyword) + 1 );

	    if (len2 > 0)
		space_needed -= len2 + 1;
	
	    VERB_PRINT2("Removing append option '%s'\n", keyword);
	
	    make_space(&buffer, size, pos2 - strlen(keyword) - 2 - space_needed, space_needed);
	} else {
	    VERB_PRINT("option not found\n");
	}
    } else {
	// if new option, insert keyword first
	if (!found) {
	    space_needed = strlen(keyword) + 1;
	    make_space(&buffer, size, pos2 + 1 - space_needed, space_needed);

	    for (i = 0; i < strlen(keyword); i++) 
		pos2[i] = keyword[i];


	    pos2[i] = ' ';

	    pos2 += strlen(keyword);
	    len2 = 0;
	    found = 1;
	}
	
	strncpy(old_val, pos2, len2);
	old_val[len2] = '\0';
	
	if (strlen(value) != len2) {
	    space_needed = strlen(value) - len2;
	    
	    // setting an existing option w/o and value to a value
	    if (len2 == 0)
		space_needed++;
	    
	    // setting option with no value
	    if (strlen(value) == 0) {
		space_needed--;
		pos2--;
		pos2[0] = ' ';
	    }
	    
	    make_space(&buffer, size, pos2 + 1 - space_needed, space_needed);
	} 	    
	
	VERB_PRINT4("Setting value for append option '%s' from '%s' to '%s'\n", keyword, old_val, value);

	if (strlen(value) > 0) {
	    // setting an existing option w/o and value to a value
	    if (len2 == 0) {
		pos2[0] = '=';
		
		for (i = 0; i < strlen(value); i++) 
		    pos2[i+1] = value[i];
	    } else {
		for (i = 0; i < strlen(value); i++) 
		    pos2[i] = value[i];
	    }
	}
    }
    
    if (found) {
	VERB_PRINT("Result:\n");
	show_buffer(buffer, size);
	
	dir_entry = get_dir_entry(SYSLINUX_CFG_FILE, boot_floppy);
	
	if (write_file(fh, boot_floppy, dir_entry, buffer) == 0) {
		    fprintf(stderr, "Error writing file. Aborting\n");
	    return 0;
	}
    } 
       
    return found;   
}

// do_command_line
//   cmd: 1: cmd_get_syslinux_opt
//        2: cmd_set_syslinux_opt
//        4: cmd_get_append_opt
//        8: cmd_set_append_opt
//        other: invalid
//
//   returns: 0 OK
//            1 CANT READ OPTIONS
//            2 TARGET NOT FOUND
//            3 NO VALUE EXPECTED
//            4 SYSLINUX OPT NOT FOUND
//            5 APPEND OPT NOT FOUND
//            6 
//            7 
//            8 INTERNAL ERROR (shouldn't happen)
//            9 NOT IMPLEMENTED

int do_commandline(FILE *fh, struct boot_floppy_t *boot_floppy, int cmd, int opt_target, const unsigned char *target) {
    int size;
    int len;
    int status = 0;
    int parse_result;
    int remove_opt = 0;
    unsigned char *buffer;
    unsigned char commandline[COMMAND_LINE_LENGTH + 1];
    struct dir_entry_t dir_entry;
    unsigned char *pos = NULL;
    unsigned char keyword[COMMAND_LINE_LENGTH + 1];
    unsigned char value[COMMAND_LINE_LENGTH + 1];

    if (opt_target)
	VERB_PRINT2("Target: [%s]\n", target);

    if (! (cmd == 1 || cmd == 2 || cmd == 4 || cmd == 8) )
	return 8;
    
	// get command line from stdin
	if (!read_commandline(commandline))
	    return 2;

	VERB_PRINT2("Commandline: [%s]\n", commandline);

	parse_result = parse_command(commandline, keyword, value);

	if (parse_result == 0)
	    return 1;

	// with get_* only allow keyword
	if (cmd == 1 || cmd == 4)
	    if (parse_result != 1)
		return 3;
	
	// with set_* allow keyword or keyword + value
	if (cmd == 2 || cmd == 8)
	    if (parse_result != 1 && parse_result != 2)
		return 8;

	trim_spaces(keyword);	
	trim_spaces(value);
	
	// test if we have keyword
	if (strlen(keyword) == 0)
	    return 1;

	if ((cmd == 2 || cmd == 8) && parse_result == 2 && strlen(value) == 0)
	    remove_opt = 1;
	
	// now we have valid arguments
	

	if (cmd == 1 || cmd == 4) {
	    VERB_PRINT2("GET    Keyword: [%s]\n", keyword);
	} else {
	    if (parse_result == 1) {
		VERB_PRINT2("SET    Keyword: [%s]\n", keyword);
	    } else {
		if (!remove_opt) {
		    VERB_PRINT3("SET    Keyword: [%s] Value: [%s]\n", keyword, value);
		} else {
		    VERB_PRINT2("REMOVE Keyword: [%s]\n", keyword);
		}
	    }
	}
	
	dir_entry = get_dir_entry(SYSLINUX_CFG_FILE, boot_floppy);
	
	// allocate read_buffer for reading file
	if ((buffer = malloc((size_t) (dir_entry.size))) == NULL) {
	    fprintf(stderr, "Error mallocing buffer for reading file\n");
	    return 6;
	}

	if (read_file(fh, boot_floppy, dir_entry, buffer) == 0) {
	    fprintf(stderr, "Error reading file. Aborting\n");
	    return 7;
	}

	size = dir_entry.size;
	if (!find_target(opt_target, target, buffer, size, &pos, &len))
	    return 2;

	if (opt_verbose) {
	    fprintf(stderr, "Target");
	    if (opt_target) {
		VERB_PRINT2(" [%s]\n", target);
	    } else {
		VERB_PRINT(" [DEFAULT]\n");
	    }

	    show_buffer(pos, len);
	}

	switch(cmd) {
	case 1:
	    VERB_PRINT("Action: get_syslinux_opt\n");
	    if (!get_syslinux_opt(keyword, pos, len))
		status = 4;
	    break;

	case 2:
	    VERB_PRINT("Action: set_syslinux_opt\n");
	    if (!set_syslinux_opt(fh, boot_floppy, keyword, value, remove_opt, buffer, size, pos, len))
		status = 1; // change
	    break;
	    
	case 4:
	    VERB_PRINT("Action: get_append_opt\n");
	    if (!get_append_opt(keyword, pos, len))
		status = 5;
	    break;

	case 8:
	    VERB_PRINT("Action: set_append_opt\n");
	    if (!set_append_opt(fh, boot_floppy, keyword, value, remove_opt, buffer, size, pos, len))
		status = 1; // change
	    status = 1; // change
	    break;

	default:
	    return 8;
	    break;
	}

	free(buffer);
	
	return status;
}
	

void show_help(char *name) {
    fprintf(stderr, "%s %s\n  written by %s\n\n", KNOPPIX_CUSTOMIZE_NAME,
                                                  KNOPPIX_CUSTOMIZE_VERSION,
                                                  KNOPPIX_CUSTOMIZE_AUTHOR);
    fprintf(stderr, "\n");
    fprintf(stderr, "Please note this software is still in developement. and there\n");
    fprintf(stderr, "are still some serious limitations.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Usage: %s [OPTION]\n", name);
    fprintf(stderr, "      --action ACTION            action to perform. Mandatory!\n"); 
    fprintf(stderr, "               list              list files in boot image\n");
    fprintf(stderr, "               export_file       export file from image to localfile\n");
    fprintf(stderr, "               import_file       import local file into image\n");
    fprintf(stderr, "               export_floppy     export boot floppy from image to localfile\n");
    fprintf(stderr, "               import_floppy     import boot floppy from local file into image\n");

    fprintf(stderr, "               get_syslinux_opt  get the value for a syslinux option\n");
    fprintf(stderr, "               set_syslinux_opt  set the value for a syslinux option\n");
    fprintf(stderr, "               get_append_opt    get the value for an APPEND-option\n");
    fprintf(stderr, "               set_append_opt    set the value for an APPEND-option.\n");

    fprintf(stderr, "\n");
    fprintf(stderr, "      --image KNOPPIX-Image-File the Knoppix image to customize. Mandatory!\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "      --local_file filename      filepath on local filesystem.\n");
    fprintf(stderr, "                                 Mandatory with --action export_* and import_*\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "      --image_file filename      file in the image (maybe . when importing)'\n");
    fprintf(stderr, "                                 Mandatory with --action export_file and import_file\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "      --target boottarget        syslinux.cfg boot target to operate on\n");
    fprintf(stderr, "                                 defaults to DEFAULT. Only used with get_* and set_*.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "      --verbose                  be verbose\n"); 
    fprintf(stderr, "\n");
    fprintf(stderr, "      --version                  show version\n"); 
    fprintf(stderr, "\n");
    fprintf(stderr, "For details please read manpage knoppix-customize(1).\n");
    fprintf(stderr, "\n");
}

void show_version() {
    printf("%s %s\n", KNOPPIX_CUSTOMIZE_NAME, KNOPPIX_CUSTOMIZE_VERSION);
}

int main(int argc, char **argv) {
    char knoppix_image_filename[PATH_MAX];
    unsigned char target[FAT_FILENAME_LENGTH + 1];
    int exit_status = EXIT_OK;

    struct stat stat_buf, stat_buf2;
    FILE *fh;
    struct boot_floppy_t boot_floppy;
    int c;

    int opt_help             = 0;
    int opt_version          = 0;
    int opt_local_file       = 0;
    int opt_image_file       = 0;
    int opt_image            = 0;
    int opt_action           = 0;
    int cmd_list             = 0;
    int cmd_export_file      = 0;
    int cmd_import_file      = 0;
    int cmd_export_floppy    = 0;
    int cmd_import_floppy    = 0;
    int opt_target           = 0;
    int cmd_get_syslinux_opt = 0;
    int cmd_set_syslinux_opt = 0;
    int cmd_get_append_opt   = 0;
    int cmd_set_append_opt   = 0;
    int cmd_unknown_action   = 0;

    char local_filename[PATH_MAX];
    char image_filename[FAT_FILENAME_LENGTH + 2];

    while (1) {
	int option_index = 0;
	static struct option long_options[] = {
	    {"help",        0, 0, 0},
	    {"version",     0, 0, 0},
	    {"verbose",     0, 0, 0},
	    {"action",      1, 0, 0},
	    {"local_file",  1, 0, 0},
	    {"image_file",  1, 0, 0},
	    {"image",       1, 0, 0},
	    {"target",      1, 0, 0},
	    {0,             0, 0, 0}
	};
	
	c = getopt_long (argc, argv, "", long_options, &option_index);
	if (c == -1) 
	    break;
	
	if (c == 0) {
	    if (strncmp(long_options[option_index].name, "help", 4) == 0) {
		opt_help = 1;
		continue;
	    }

	    if (strncmp(long_options[option_index].name, "version", 7) == 0) {
		opt_version = 1;
		continue;
	    }

	    if (strncmp(long_options[option_index].name, "verbose", 7) == 0) {
		opt_verbose = 1;
		continue;
	    }

	    if (strncmp(long_options[option_index].name, "local_file", 10) == 0) {
		opt_local_file = 1;
		strncpy(local_filename, optarg, PATH_MAX);
		continue;
	    }

	    if (strncmp(long_options[option_index].name, "image_file", 10) == 0) {
		opt_image_file = 1;
		strncpy(image_filename, optarg, FAT_FILENAME_LENGTH + 1);
		continue;
	    }

	    if (strncmp(long_options[option_index].name, "image", 5) == 0) {
		opt_image = 1;
		strncpy(knoppix_image_filename, optarg, PATH_MAX);
		continue;
	    }

	    if (strncmp(long_options[option_index].name, "action", 6) == 0) {
		opt_action = 1;
		
		if (strncmp(optarg, "list", 4) == 0)
		    cmd_list = 1;
		else if (strncmp(optarg, "export_file", 10) == 0)
		    cmd_export_file = 1;
		else if (strncmp(optarg, "import_file", 10) == 0)
		    cmd_import_file = 1;
		else if (strncmp(optarg, "import_floppy", 12) == 0)
		    cmd_import_floppy = 1;
		else if (strncmp(optarg, "export_floppy", 12) == 0)
		    cmd_export_floppy = 1;

		else if (strncmp(optarg, "get_syslinux_opt", 15) == 0)
		    cmd_get_syslinux_opt = 1;
		else if (strncmp(optarg, "set_syslinux_opt", 15) == 0)
		    cmd_set_syslinux_opt = 1;
		else if (strncmp(optarg, "get_append_opt", 13) == 0)
		    cmd_get_append_opt = 1;
		else if (strncmp(optarg, "set_append_opt", 13) == 0)
		    cmd_set_append_opt = 1;

		else
		    cmd_unknown_action = 1;

		continue;
	    }

	    if (strncmp(long_options[option_index].name, "target", 6) == 0) {
		opt_target = 1;
		strncpy(target, optarg, FAT_FILENAME_LENGTH);
		continue;
	    }
	} 
    }

    // there should be no further arguments
    if (optind < argc) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      Unknown arguments: ");
	while (optind < argc)
                   fprintf(stderr, "%s ", argv[optind++]);
	fprintf(stderr, "\n");
	fprintf(stderr, "      try --help for more information\n");

	exit(EXIT_WRONG_ARGUMENTS);
    }

    // show help
    if (opt_help) {
	show_help(basename(argv[0]));
	exit(EXIT_OK);
    }

    // show version
    if (opt_version) {
	show_version();
	exit(EXIT_OK);
    }

    // --image and --action are mandatory
    if (cmd_unknown_action) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      unknown action\n");
	fprintf(stderr, "      try --help for more information\n");
	exit(EXIT_WRONG_ARGUMENTS);
    }

    // --image and --action are mandatory
    if (! opt_image || ! opt_action) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      --image and --action are mandatory\n");
	fprintf(stderr, "      try --help for more information\n");
	exit(EXIT_WRONG_ARGUMENTS);
    }

    // --local_file and image_file are mandatory for --export_file and --import_file
    if ((cmd_export_file || cmd_import_file) && ! (opt_local_file && opt_image_file)) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      --local_file and --image_file are mandatory with\n");
	fprintf(stderr, "      --action export_file and --action import_file\n");
	fprintf(stderr, "      try --help for more information\n");
	exit(EXIT_WRONG_ARGUMENTS);
    }

    // --local_file is mandatory for --export_floppy and --import_floppy
    if ((cmd_export_floppy || cmd_import_floppy) && ! opt_local_file) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      --local_file is mandatory with\n");
	fprintf(stderr, "      --action export_floppy and --action import_floppy\n");
	fprintf(stderr, "      try --help for more information\n");
	exit(EXIT_WRONG_ARGUMENTS);
    }

    // --target can only be used with get_* and set_*
    if (opt_target && ! (cmd_get_syslinux_opt || cmd_set_syslinux_opt || cmd_get_append_opt || cmd_set_append_opt)) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      --target can only be used with get_* and set_*\n");
	fprintf(stderr, "      try --help for more information\n");
	exit(EXIT_WRONG_ARGUMENTS);
    }

    // --image_file can only be used with export_* and import_*\n
    if (opt_image_file &&! (cmd_export_floppy || cmd_import_floppy || cmd_export_file || cmd_import_file)) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      --image_file can only be used with export_* and import_*\n");
	fprintf(stderr, "      try --help for more information\n");
	exit(EXIT_WRONG_ARGUMENTS);
    }

    // --local_file can only be used with export_* and import_*\n
    if (opt_local_file &&! (cmd_export_floppy || cmd_import_floppy || cmd_export_file || cmd_import_file)) {
	fprintf(stderr, "Usage: %s [OPTION]\n", argv[0]);
	fprintf(stderr, "      --local_file can only be used with export_* and import_*\n");
	fprintf(stderr, "      try --help for more information\n");
	exit(EXIT_WRONG_ARGUMENTS);
    }

    // test if image exists
    if (stat(knoppix_image_filename, &stat_buf) != 0) {
        fprintf(stderr, "Error: Could not stat image file '%s'\n", knoppix_image_filename);
        exit(EXIT_CANT_STAT_IMAGE);
    }

    // test if local_file exists
    if (cmd_import_file || cmd_import_floppy) {
	if (!strncmp(local_filename, "-", 1) == 0)
	    if (stat(local_filename, &stat_buf2) != 0) {
		fprintf(stderr, "Error: Could not stat local file '%s'\n", local_filename);
		exit(EXIT_CANT_STAT_LOCAL_FILE);
	    }
    }

    // open the image file for reading and writing
    if (cmd_import_file || cmd_import_floppy || cmd_set_syslinux_opt || cmd_set_append_opt) {
	if ((fh = fopen(knoppix_image_filename, "rb+")) == NULL) {
	    fprintf(stderr, "Error opening image '%s' for read/write\n", knoppix_image_filename);
	    exit(EXIT_CANT_READ_IMAGE);
	}
    } else {
	if ((fh = fopen(knoppix_image_filename, "rb")) == NULL) {
	    fprintf(stderr, "Error opening image '%s' for read\n", knoppix_image_filename);
	    exit(EXIT_CANT_READ_IMAGE);
	}
    }

    VERB_PRINT3("%s %s\n", KNOPPIX_CUSTOMIZE_NAME, KNOPPIX_CUSTOMIZE_VERSION);

    // find out if it's ISO or Floppy-image, in ISO find the floppy image
    if (find_boot_image(fh, stat_buf, &boot_floppy) == 0) {
        fprintf(stderr, "Error reading image. Aborting\n");
        fclose(fh);
        exit(EXIT_CANT_READ_IMAGE);
    }

    // Get the boot floppy's root directory
    if (get_fat_root_dir(fh, &boot_floppy) == 0) {
        fprintf(stderr, "Error parsing boot image. Aborting\n");
        fclose(fh);
        exit(EXIT_CANT_PARSE_IMAGE);
    }

    if (opt_verbose)
	show_root_dir(&boot_floppy);

    // test if image_file exists
    // can only test this after reading the image

    // if image-file is "." use file name from local_filename (without directory)
    if (cmd_import_file && (strncmp(image_filename, ".", 1) == 0)) {
	strncpy(image_filename, basename(local_filename), FAT_FILENAME_LENGTH + 1);
    }

    if ((cmd_import_file || cmd_export_file) && !file_exists_in_image(image_filename, &boot_floppy)) {
        fprintf(stderr, "Error: image_file '%s' not in image\n", image_filename);
        exit(EXIT_CANT_PARSE_IMAGE);
    }

    // list files on boot floppy
    if (cmd_list)
	list_files(&boot_floppy);

    // export file from image to filesystem
    if (cmd_export_file) {
	if (!strncmp(local_filename, "-", 1) == 0)
	    if (stat(local_filename, &stat_buf2) == 0) {
		if (S_ISDIR(stat_buf2.st_mode)) {
		    if (local_filename[strlen(local_filename)-1] != G_DIR_SEPARATOR)
			strncat(local_filename, G_DIR_SEPARATOR_S, PATH_MAX);
		    strncat(local_filename, image_filename, PATH_MAX);
		}
	    } 
	
	if(export_to_local_file(fh, &boot_floppy, image_filename, local_filename) == 0) {
	    fprintf(stderr, "Error exporting file to local file. Aborting\n");
	    fclose(fh);
	    exit(EXIT_CANT_EXPORT);
	}
    }
	
    // import file from filesystem into image
    if (cmd_import_file)
	if(import_from_local_file(fh, &boot_floppy, image_filename, local_filename) == 0) {
	    fprintf(stderr, "Error importing file from local file and writing to image. Aborting\n"); 
	    fclose(fh);
	    exit(EXIT_CANT_IMPORT);
	}

    // export boot floppy from image to filesystem
    if (cmd_export_floppy)
	if(export_floppy(fh, &boot_floppy, local_filename) == 0) {
	    fprintf(stderr, "Error exporting boot floppy to local file. Aborting\n"); 
	    fclose(fh);
	    exit(EXIT_CANT_EXPORT);
	}

    // import boot floppy from filesystem into image
    if (cmd_import_floppy)
	if(import_floppy(fh, &boot_floppy, local_filename) == 0) {
	    fprintf(stderr, "Error importing boot floppy and writing to image. Aborting\n"); 
	    fclose(fh);
	    exit(EXIT_CANT_IMPORT);
	}
    
    if (cmd_get_syslinux_opt || cmd_set_syslinux_opt || cmd_get_append_opt || cmd_set_append_opt) {
	switch(do_commandline(fh, &boot_floppy,
		 	      cmd_get_syslinux_opt       +
			     (cmd_set_syslinux_opt << 1) + 
			     (cmd_get_append_opt   << 2) + 
			     (cmd_set_append_opt << 3),
			     opt_target, target)) {


//   returns: 0 OK
//            1 INTERNAL ERROR
//            2 CANT READ OPTIONS
//            3 TARGET NOT FOUND
//            4 NO VALUE EXPECTED
//            5 SYSLINUX OPT NOT FOUND
//            6 APPEND OPT NOT FOUND

	case 0:
	    exit_status = EXIT_OK;
	    break;
	case 1:
	    exit_status = EXIT_CANT_READ_OPTIONS;
	    break;
	case 2:
	    exit_status = EXIT_TARGET_NOT_FOUND;
	    break;
	case 3:
	    exit_status = EXIT_NO_VALUE_EXPECTED;
	    break;
	case 4:
	    exit_status = EXIT_SYSLINUX_OPT_NOT_FOUND;
	    break;
	case 5:
	    exit_status = EXIT_APPEND_OPT_NOT_FOUND;
	    break;
	case 6:
	    exit_status = EXIT_INTERNAL_ERROR;
	    break;
	case 7:
	    exit_status = EXIT_INTERNAL_ERROR;
	    break;
	case 8:
	    exit_status = EXIT_INTERNAL_ERROR;
	    break;
	case 9:
	    exit_status = EXIT_NOT_IMPLEMENTED;
	    fprintf(stderr, "Sorry function is not implemented yet.\n");
	    break;
	}
    }

    fclose(fh);
    
    exit(exit_status);
}

